#!/usr/bin/env python3
import json, subprocess, sys
from pathlib import Path

def run(cmd):
    p = subprocess.run(cmd, capture_output=True, text=True)
    return p.returncode, p.stdout, p.stderr

def J(p): return json.loads(Path(p).read_text(encoding='utf-8'))

def main():
    root = Path.cwd()
    eng  = root/'engine_d9_eproxy_v4.py'
    conf = root/'config'
    outC = root/'outputs'/'coarse'; outF = root/'outputs'/'fine'
    outC.mkdir(parents=True, exist_ok=True); outF.mkdir(parents=True, exist_ok=True)
    (root/'logs').mkdir(parents=True, exist_ok=True)

    # coarse
    rc,so,se = run([sys.executable, str(eng),
                    '--manifest', str(conf/'manifest_coarse.json'),
                    '--diag',     str(conf/'diagnostics.json'),
                    '--out',      str(outC)])
    (root/'logs'/'engine_stdout_coarse.txt').write_text(so)
    (root/'logs'/'engine_stderr_coarse.txt').write_text(se)

    # fine
    rc,so,se = run([sys.executable, str(eng),
                    '--manifest', str(conf/'manifest_fine.json'),
                    '--diag',     str(conf/'diagnostics.json'),
                    '--out',      str(outF)])
    (root/'logs'/'engine_stdout_fine.txt').write_text(so)
    (root/'logs'/'engine_stderr_fine.txt').write_text(se)

    aC = J(outC/'audits'/'em_eproxy.json')
    aF = J(outF/'audits'/'em_eproxy.json')
    sC = J(outC/'metrics'/'slope_fit.json')
    sF = J(outF/'metrics'/'slope_fit.json')

    tau = 0.10
    dsl = abs((sC.get('slope') or 0.0) - (sF.get('slope') or 0.0))
    result = {
        "coarse_PASS": aC.get('PASS', False),
        "fine_PASS":   aF.get('PASS', False),
        "slope_coarse": sC.get('slope'),
        "slope_fine":   sF.get('slope'),
        "slope_diff":   dsl,
        "tau_mesh":     tau,
        "MESH_PASS": bool(aC.get('PASS',False) and aF.get('PASS',False) and dsl <= tau)
    }
    (root/'outputs'/'mesh_summary.json').write_text(json.dumps(result, indent=2))
    print("D9 DUAL SUMMARY (v4):", json.dumps(result))

if __name__ == '__main__':
    main()
